#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <unistd.h>
#include <string.h>

#include "main.h"
#include "types.h"
#include "arg.h"
#include "ipspace.h"
#include "cvt.h"


static
void
usage(char *problem, ...)
{
	va_list ap;
	char buf[256];

	fprintf(
		stderr,
		USAGE_TITLE "\n"
		USAGE_CODER "\n"
		"edit: " USAGE_EDIT "\n"
		"comp: " USAGE_COMP "\n"
		"\n"
		"Options:\n"
		"   + d/directive    base_ip:bits:repeat[:key[:offset_start[:offset_end]]]\n"
		"     f/find         ip\n"
		"     s/split        number\n"
		"\n"
		"Examples:\n"
		"   d=192.168.1.0/24:0              | 192.168.1.*, don't repeat\n"
		"   d=192.168.1.0/24:0:0xdead s=4   | display 192.168.1.* using key 0xdead divided into 4 parts\n"
		"   d=24.0.0.0/8:1                  | 24.*, repeat\n"
		"\n"
		"If there are no arguments, directives are accepted on stdin, one per line.\n"
	);

	if (problem) {
		va_start(ap, problem);
		vsnprintf(buf, sizeof(buf), problem, ap);
		va_end(ap);

		fprintf(stderr, "problem: %s\n\n", buf);
	}
}


int
main(int argc, char **argv)
{
	s8 buf[64], *p;
	u32 opts,
		args_expected, args_handled,
		searchip, numsplit, offset;
	ipspace ips;

	if (argc == 1) {
		while (fgets(buf, sizeof(buf), stdin)) {
			if ((p = strchr(buf, '\n'))) {
				*p = '\0';
			}

			if (ipspace_init(&ips, buf)) {
				ipspace_generate(&ips, STDOUT_FILENO);
			}
		}
		return EXIT_SUCCESS;
	}

	opts = 0;
	args_expected = argc - 1;
	args_handled = 0;

	/* -- -- -- -- */

	/* directive is mandatory */
	if ((p = arg_get(argc, argv, "d")) || (p = arg_get(argc, argv, "directive"))) {
		if (!ipspace_init(&ips, p)) {
			usage("invalid ipspace directive");
			return EXIT_FAILURE;
		}
		args_handled++;
		opts |= OPT_DIRECTIVE;
	}

	/* user can specify find or split, but shouldn't do both */
	if ((p = arg_get(argc, argv, "f")) || (p = arg_get(argc, argv, "find"))) {
		if (!cvt_csip_u32(&searchip, p)) {
			usage("invalid search IP");
			return EXIT_FAILURE;
		}
		args_handled++;
		opts |= OPT_FINDOFFSET;
	} else {
		if ((p = arg_get(argc, argv, "s")) || (p = arg_get(argc, argv, "split"))) {
			if (!cvt_cs_u32(&numsplit, p)) {
				usage("invalid split number");
				return EXIT_FAILURE;
			}
			args_handled++;
			opts |= OPT_SPLIT;
		}
	}

	/* -- -- -- -- */

	if (args_expected != args_handled) {
		usage(NULL);
		return EXIT_FAILURE;
	}

	if (!(opts & OPT_DIRECTIVE)) {
		usage("must specify ipspace directive");
		return EXIT_FAILURE;
	}

	/* -- -- -- -- */

	if (opts & OPT_FINDOFFSET) {
		if (!ips.key_specified) {
			usage("specify a key if you want to use find");
			return EXIT_FAILURE;
		}
		ipspace_find_offset(&ips, searchip, &offset);
		printf("%#x\n", offset);
		return EXIT_SUCCESS;
	}

	if (opts & OPT_SPLIT) {
		if (!ips.key_specified) {
			usage("specify a key if you want to use split");
			return EXIT_FAILURE;
		}
		ipspace_print_split(&ips, numsplit);
		return EXIT_SUCCESS;
	}

	/* -- -- -- -- */

	if (!ipspace_generate(&ips, STDOUT_FILENO)) {
		fprintf(stderr, "ipspace_generate() failed\n");
		return EXIT_FAILURE;
	}

	return EXIT_SUCCESS;
}
