#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <math.h>
#include <time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/stat.h>
#include <fcntl.h>

#include "ipspace.h"
#include "types.h"
#include "cvt.h"
#include "fnv.h"


static
bool
ipspace_get_random_key(u32 const *key)
{
	s32 fd;

	fd = open(IPSPACE_RANDOM_DEV, O_RDONLY);
	if (fd == -1) {
		perror("open");
		return false;
	}

	if (read(fd, (void *)key, 4) != 4) {
		perror("read");
		return false;
	}

	close(fd);
	return true;
}

/* initialize the subkeys array.
 * we use the key if one is specified.
 * if not, we read 32 bits from the random device with ipspace_get_random_key().
 */
static
bool
ipspace_init_subkeys(ipspace const *ips)
{
	u32 u,
		key,
		*subkeys = ips->subkeys;

	if (ips->key_specified) {
		key = ips->key;
	} else {
		if (!ipspace_get_random_key(&key)) return false;
	}

	/* supposedly, this is some kind of standard crypto subkeys generation procedure */
	for (u=IPSPACE_NUMROUNDS; u--; ) {
		key ^= (key >> 13);
		subkeys[u] = key = ((key << 17) + key);
	}

	return true;
}

/* initialize the ipspace struct based on the supplied cstring directive.
 * format: base_ip:bits:repeat[:key[:offset_start[:offset_end]]]
 * where : is one of the legal delimeters.
 * 		base_ip  		ip
 * 		bits     		0-32
 * 		repeat   		0 or 1
 * 		key				u32
 * 		offset_start	u32
 * 		offset_end		u32
 */
bool
ipspace_init(ipspace *ips, s8 *ipsd_cs)
{
	s8 *ipsd_cs_dup, *ipsd_cs_dup_backup, *p;
	u32 base_ip, key, offset_start, offset_end, numnbits, numhbits, hmask, numrbits;
	bool return_status, repeat, key_specified, offset_start_specified, offset_end_specified;

	ipsd_cs_dup = ipsd_cs_dup_backup = strdup(ipsd_cs);
	if (!ipsd_cs_dup) {
		perror("strdup");
		return false;
	}

	return_status = true;
	repeat = key_specified = offset_start_specified = offset_end_specified = false;

#define GET_NEXT_FIELD strsep(&ipsd_cs_dup, IPSPACE_DIRECTIVE_DELIM)

	p = GET_NEXT_FIELD;								/* base ip */
	if (!p || !cvt_csip_u32(&base_ip, p)) {
		return_status = false;
		goto end;
	}

	p = GET_NEXT_FIELD;								/* netmask (should be 0-32) */
	if (!p || !cvt_cs_u32(&numnbits, p) || numnbits > 32) {
		return_status = false;
		goto end;
	}

	p = GET_NEXT_FIELD;								/* repeat (should be 0 or 1) */
	if (!p || !cvt_cs_u32(&repeat, p) || (repeat!=0 && repeat!=1)) {
		return_status = false;
		goto end;
	}

	numhbits = 32 - numnbits;
	hmask = (numhbits == 32) ? 0xffffffff : (1 << numhbits) - 1;
	numrbits = numhbits / 2;

	p = GET_NEXT_FIELD;								/* key (should be u32) */
	if (p) {
		if (!cvt_cs_u32(&key, p)) {
			return_status = false;
			goto end;
		}
		key_specified = true;

		p = GET_NEXT_FIELD;							/* offset_start (should be u32) */
		if (p) {
			if (!cvt_cs_u32(&offset_start, p) || offset_start>hmask) {
				return_status = false;
				goto end;
			}
			offset_start_specified = true;

			p = GET_NEXT_FIELD;						/* offset_end (should be u32) */
			if (p) {
				if (!cvt_cs_u32(&offset_end, p) || offset_end>hmask || offset_end<offset_start) {
					return_status = false;
					goto end;
				}
				offset_end_specified = true;

				p = GET_NEXT_FIELD;					/* there shouldn't be any more fields */
				if (p) {
					return_status = false;
					goto end;
				}
			}
		}
	}

	ips->base_ip = ntohl(base_ip);	/* cvt_csip_u32() returned NBO, but we need native order. */
	ips->repeat = repeat;

	if (key_specified) {
		ips->key_specified = true;
		ips->key = key;
	} else {
		ips->key_specified = false;
	}

	if (!ipspace_init_subkeys(ips)) {
		return_status = false;
		goto end;
	}

	ips->offset_start = (offset_start_specified ? offset_start : 0);	/* offset_start is 0 unless otherwise specified */
	ips->offset_end = (offset_end_specified ? offset_end : hmask);		/* offset_end is the end of the range unless otherwise specified */
	ips->hmask = hmask;
	ips->rmask = (1 << numrbits) - 1;
	ips->numrbits = numrbits;

	end:
	free(ipsd_cs_dup_backup);
	return return_status;
}

/* traverse the range writing 4-byte NBO IP records to the specified file descriptor.
 */
bool
ipspace_generate(ipspace * const ips, const int fd)
{
	u32 n			= ips->offset_start,
		oe 			= ips->offset_end,
		base_ip 	= ips->base_ip,
		rmask		= ips->rmask,
		*subkeys 	= ips->subkeys,
		numrbits 	= ips->numrbits,
		round, u;
	bool repeat		= ips->repeat;
	u16	rblk, lblk, tblk;

	repeat:

	if (!ips->key_specified) {		/* it's useless to reinitialize the subkeys if there is a user key */
		if (!ipspace_init_subkeys(ips)) return false;
	}

	do {
		rblk = n & rmask;
		lblk = n >> numrbits;
		for (round=IPSPACE_NUMROUNDS; round--; ) {
			tblk = rblk;
			rblk = (fnv32_u16_subkey(rblk, subkeys[round]) & rmask) ^ lblk;
			lblk = tblk;
		}

		u = htonl(base_ip | (lblk << numrbits) | rblk);
		write(fd, (void *)&u, 4);
	} while (n++ != oe);

	if (repeat) goto repeat;
	return true;
}


/* given an ip, determine the corresponding offset.
 */
void
ipspace_find_offset(ipspace * const ips, const u32 ip, u32 *offset) {
	u32	rmask		= ips->rmask,
		*subkeys 	= ips->subkeys,
		numrbits 	= ips->numrbits,
		hmask		= ips->hmask,
		round, c;
	u16	rblk, lblk, tblk;

	c = ntohl(ip) & hmask;

	rblk = c & rmask;
	lblk = c >> numrbits;
	for (round=0; round<IPSPACE_NUMROUNDS; round++) {
		tblk = lblk;
		lblk = (fnv32_u16_subkey(lblk, subkeys[round]) & rmask) ^ rblk;
		rblk = tblk;
	}

	*offset = (lblk << numrbits) | rblk;
}


/* print ipspace directives split up.
 * this function assumes that a key has been specified.
 */
void
ipspace_print_split(ipspace * const ips, const u32 numsplit)
{
	u32 i, a, b,
		numnbits,
		numtodo, each,
		base_ip      = htonl(ips->base_ip),
		repeat       = ips->repeat,
		key          = ips->key,
		offset_start = ips->offset_start,
		offset_end   = ips->offset_end;
	u8 ip_cs[32];

	numnbits = 32 - (log(ips->hmask + 1) / log(2));		/* /24 means hostmask is 0xff.  log base 2 of 256 = 8.  32 - 8 = 24 */

	numtodo = offset_end - offset_start + 1;	/* 0xff - 0 means 256 to do */
	if (numsplit > numtodo) return;				/* catch stupid stuff like dividing 8 into 10 parts */
	each = numtodo / numsplit;

	inet_ntop(AF_INET, &base_ip, ip_cs, sizeof(ip_cs));

	for (i=0; i<numsplit-1; i++) {
		a = offset_start + (i*each);
		b = a + each - 1;
		printf("%s/%u:%u:%#x:%#x:%#x\n", ip_cs, numnbits, repeat, key, a, b);
	}

	/* the last line is weird because it must absorb any extra stuff
	 * like in the case of uneven division.
	 * like 0-9 in 4 parts: 0-1, 2-3, 4-5, 6-9
	 */
	printf("%s/%u:%u:%#x:%#x:%#x\n", ip_cs, numnbits, repeat, key, offset_start + (i * each), offset_end);
}
